﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using RestCarService.Models;
using RestCarService.Infrastructure;

namespace RestCarService.Controllers
{
  public class CarsController : Controller
  {
    private ICarsRepository repository;

    public CarsController(ICarsRepository repository)
    {
      this.repository = repository;
    }

    // Cars/{make}/{maxYear}/{maxMileage}
    // Cars/{make}/{model}/{maxYear}/{maxMileage}
    // Cars/{make}/{model}/{type}/{minYear}/{maxMileage}
    // Cars/{make}/{model}/{color}
    [HttpGet]
    public ActionResult List(
      string make = "All",
      string model = "All",
      string type = "All",
      string color = "All",
      int minYear = 0,
      string maxMileage = "0km")
    {
      var carsQuery = repository.Cars;

      if (make != "All")
      {
        carsQuery = carsQuery.Where(c => c.Make == make);
      }
      if (model != "All")
      {
        carsQuery = carsQuery.Where(c => c.Model == model);
      }
      if (type != "All")
      {
        carsQuery = carsQuery.Where(c => c.Type == type);
      }
      if (color != "All")
      {
        carsQuery.Where(c => c.Color == color);
      }
      if (minYear > 0)
      {
        carsQuery = carsQuery.Where(c => c.Year >= minYear);
      }
      int mileage = 0;
      if (maxMileage.EndsWith("km"))
      {
        maxMileage = maxMileage.TrimEnd('k', 'm');
        mileage = Convert.ToInt32(maxMileage);
      }
      if (maxMileage.EndsWith("mi"))
      {
        maxMileage = maxMileage.TrimEnd('m', 'i');
        mileage = Convert.ToInt32(maxMileage);
      }
      if (mileage > 0)
      {
        carsQuery = carsQuery.Where(c => c.Mileage <= mileage);
      }

      return View(carsQuery.ToList());
    }

    [HttpGet]
    [ActionName("Single")]
    public ActionResult Details(int id)
    {
      var car = repository.Cars.Where(c => c.Id == id).SingleOrDefault();
      if (null == car)
      {
        return HttpNotFound();
      }
      return View("Details", car);
    }

    [ChildActionOnly] // helper that prevents going to the db again
    public ActionResult Details(Car car)
    {
      return View("Details", car);
    }

    [HttpGet]
    public ActionResult Create()
    {
      return View();
    }

    [HttpPost]
    [ActionName("Single")]
    public ActionResult Create(Car car)
    {
      if (ModelState.IsValid)
      {
        repository.InsertCar(car);

        return new RestResult(Details(car),
          201, "Created", Url.Action("Single", new { id = car.Id }));
      }
      return View();
    }

    [HttpGet]
    public ActionResult Edit(int id)
    {
      var car = repository.Cars.Where(c => c.Id == id).SingleOrDefault();
      if (null == car)
      {
        return HttpNotFound();
      }
      return View(car);
    }

    [HttpPut]
    [ActionName("Single")]
    public ActionResult Edit(Car car)
    {
      if (ModelState.IsValid)
      {
        try
        {
          repository.UpdateCar(car);
        }
        catch (Exception e)
        {
          return new RestResult(View("Edit"), 404, "Not Found");
        }
        return new RestResult(Details(car),
          201, "Created", Url.Action("Details", new { id = car.Id }));
      }
      return new RestResult(View("Edit"), 415, "Unsupported Media Type");
    }

    [HttpGet]
    public ActionResult Delete(int id)
    {
      var car = repository.Cars.Where(c => c.Id == id).SingleOrDefault();
      if (null == car)
      {
        return HttpNotFound();
      }
      return View(car);
    }

    [HttpDelete]
    [ActionName("Single")]
    public ActionResult DeleteConfirmation(int id)
    {
      var car = repository.Cars.Where(c => c.Id == id).SingleOrDefault();
      if (null == car)
      {
        return HttpNotFound();
      }

      repository.DeleteCar(id);
      return new RestResult(DeleteConfirmed(car), 200, "OK");
    }

    [ChildActionOnly]
    public ActionResult DeleteConfirmed(Car car)
    {
      return View("DeleteConfirmed", car);
    }
  }
}
