﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Web.Mvc;
using System.Text.RegularExpressions;

namespace RestCarService.Infrastructure
{
  public class RestResult : ActionResult
  {
    private readonly static string[] _jsonTypes = { "application/json", "text/json" };
    private readonly static string[] _xmlTypes = { "application/xml", "text/xml" };
    private readonly static string[] _htmlTypes = { "application/xhtml", "text/html" };
    private readonly static string _formEncodedType = "application/x-www-form-urlencoded";
    private readonly static Regex _typeRegEx = new Regex(@"[\w\*]+/[\w\*]+");

    private ActionResult actionResult;

    public RestResult(ActionResult actionResult, int statusCode, string statusDescription = null, string location = null)
    {
      if (null == actionResult) throw new ArgumentException("RestResult should always wrap an actionResult", "actionResult");
      this.actionResult = actionResult;
      StatusCode = statusCode;
      StatusDescription = statusDescription;
      Location = location;
    }


    public string Location { get; private set; }

    public int StatusCode { get; private set; }

    public string StatusDescription { get; private set; }

    public override void ExecuteResult(ControllerContext context)
    {
      if (null == context)
      {
        throw new ArgumentNullException("context");
      }
      var httpContext = context.HttpContext;

      httpContext.Response.StatusCode = this.StatusCode;
      if (null != this.StatusDescription)
      {
        httpContext.Response.StatusDescription = this.StatusDescription;
      }
      if (null != this.Location)
      {
        httpContext.Response.AddHeader("Location", this.Location);
      }


      if (typeof(RedirectToRouteResult).IsInstanceOfType(actionResult))
      {
        actionResult.ExecuteResult(context);
        return;
      }

      var acceptTypes = (httpContext.Request.AcceptTypes ?? new[] { "text/html" })
          .Select(a => _typeRegEx.Match(a).Value).ToArray();

      var model = context.Controller.ViewData.Model;

      if (_jsonTypes.Any(type => acceptTypes.Contains(type)))
      {
        actionResult = new JsonResult() { Data = model, JsonRequestBehavior = JsonRequestBehavior.AllowGet };
      }
      else if (_xmlTypes.Any(type => acceptTypes.Contains(type)))
      {
        actionResult = new XmlResult() { Data = model };
      }

      actionResult.ExecuteResult(context);
    }
  }
}