/********************************************************************************
 *   This file is part of NRtfTree Library.
 *
 *   NRtfTree Library is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Lesser General Public License as published by
 *   the Free Software Foundation; either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   NRtfTree Library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU Lesser General Public License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License
 *   along with this program. If not, see <http://www.gnu.org/licenses/>.
 ********************************************************************************/

/********************************************************************************
 * Library:		NRtfTree
 * Version:     v0.3.0
 * Date:		01/05/2009
 * Copyright:   2006-2009 Salvador Gomez
 * E-mail:      sgoliver.net@gmail.com
 * Home Page:	http://www.sgoliver.net
 * SF Project:	http://nrtftree.sourceforge.net
 *				http://sourceforge.net/projects/nrtftree
 * Class:		RtfDocument
 * Description:	Clase para la generacin de documentos RTF.
 * ******************************************************************************/

using System;
using System.Collections.Generic;
using System.Text;
using System.Drawing;
using System.Globalization;
using System.IO;
using Net.Sgoliver.NRtfTree.Core;

namespace Net.Sgoliver.NRtfTree
{
    namespace Util
    {
        /// <summary>
        /// Clase para la generacin de documentos RTF.
        /// </summary>
        public class RtfDocument
        {
            #region Atributos privados

            /// <summary>
            /// Ruta del fichero a generar.
            /// </summary>
            private string path;

            /// <summary>
            /// Codificacin del documento.
            /// </summary>
            private Encoding encoding;

            /// <summary>
            /// Tabla de fuentes del documento.
            /// </summary>
            private RtfFontTable fontTable;

            /// <summary>
            /// Tabla de colores del documento.
            /// </summary>
            private RtfColorTable colorTable;

            /// <summary>
            /// rbol RTF del documento.
            /// </summary>
            private RtfTree tree;

            /// <summary>
            /// Grupo principal del documento.
            /// </summary>
            private RtfTreeNode mainGroup;

            /// <summary>
            /// Formato actual de caracter.
            /// </summary>
            private RtfCharFormat currentFormat;

            /// <summary>
            /// Formato actual de prrafo.
            /// </summary>
            private RtfParFormat currentParFormat;

            /// <summary>
            /// Formato del documento.
            /// </summary>
            private RtfDocumentFormat docFormat;

            #endregion

            #region Constructores

            /// <summary>
            /// Constructor de la clase RtfDocument.
            /// </summary>
            /// <param name="path">Ruta del fichero a generar.</param>
            /// <param name="enc">Codificacin del documento a generar.</param>
            public RtfDocument(string path, Encoding enc)
            {
                this.path = path;
                this.encoding = enc;

                fontTable = new RtfFontTable();
                fontTable.AddFont("Arial");  //Default font

                colorTable = new RtfColorTable();
                colorTable.AddColor(Color.Black);  //Default color

                currentFormat = new RtfCharFormat();
                currentParFormat = new RtfParFormat();
                docFormat = new RtfDocumentFormat();

                tree = new RtfTree();
                mainGroup = new RtfTreeNode(RtfNodeType.Group);

                InitializeTree();
            }

            /// <summary>
            /// Constructor de la clase RtfDocument. Se utilizar la codificacin por defecto del sistema.
            /// </summary>
            /// <param name="path">Ruta del fichero a generar.</param>
            public RtfDocument(string path) : this(path, Encoding.Default)
            {

            }

            #endregion

            #region Metodos Publicos

            /// <summary>
            /// Cierra el documento RTF.
            /// </summary>
            public void Close()
            {
                InsertFontTable();
                InsertColorTable();
                InsertGenerator();
                InsertDocSettings();

                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "par", false, 0));
                tree.RootNode.AppendChild(mainGroup);

                tree.SaveRtf(path);
            }

            public string GetRtfString()
            {
                InsertFontTable();
                InsertColorTable();
                InsertGenerator();
                InsertDocSettings();
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "par", false, 0));
                tree.RootNode.AppendChild(mainGroup);

                return tree.Rtf;
            }

            /// <summary>
            /// Inserta un fragmento de texto en el documento con un formato de texto determinado.
            /// </summary>
            /// <param name="text">Texto a insertar.</param>
            /// <param name="format">Formato del texto a insertar.</param>
            public void AddText(string text, RtfCharFormat format)
            {
                UpdateFontTable(format);
                UpdateColorTable(format);

                UpdateCharFormat(format);

                InsertText(text);
            }

            /// <summary>
            /// Inserta un fragmento de texto en el documento con el formato de texto actual.
            /// </summary>
            /// <param name="text">Texto a insertar.</param>
            public void AddText(string text)
            {
                InsertText(text);
            }

            /// <summary>
            /// Inserta un nmero determinado de saltos de lnea en el documento.
            /// </summary>
            /// <param name="n">Nmero de saltos de lnea a insertar.</param>
            public void AddNewLine(int n)
            {
                for(int i=0; i<n; i++)
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "line", false, 0));
            }

            /// <summary>
            /// Inserta un salto de lnea en el documento.
            /// </summary>
            public void AddNewLine()
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "line", false, 0));
            }

            /// <summary>
            /// Inicia un nuevo prrafo.
            /// </summary>
            public void AddNewParagraph()
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "par", false, 0));
            }

            /// <summary>
            /// Inserta un nmero determinado de saltos de prrafo en el documento.
            /// </summary>
            /// <param name="n">Nmero de saltos de prrafo a insertar.</param>
            public void AddNewParagraph(int n)
            {
                for (int i = 0; i < n; i++)
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "par", false, 0));
            }

            /// <summary>
            /// Inicia un nuevo prrafo con el formato especificado.
            /// </summary>
            public void AddNewParagraph(RtfParFormat format)
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "par", false, 0));

                UpdateParFormat(format);
            }

            /// <summary>
            /// Inserta una imagen en el documento.
            /// </summary>
            /// <param name="path">Ruta de la imagen a insertar.</param>
            /// <param name="width">Ancho deseado de la imagen en el documento.</param>
            /// <param name="height">Alto deseado de la imagen en el documento.</param>
            public void AddImage(string path, int width, int height)
            {
                FileStream fStream = null;
                BinaryReader br = null;

                try
                {
                    byte[] data = null;

                    FileInfo fInfo = new FileInfo(path);
                    long numBytes = fInfo.Length;

                    fStream = new FileStream(path, FileMode.Open, FileAccess.Read);
                    br = new BinaryReader(fStream);

                    data = br.ReadBytes((int)numBytes);

                    StringBuilder hexdata = new StringBuilder();

                    for (int i = 0; i < data.Length; i++)
                    {
                        hexdata.Append(GetHexa(data[i]));
                    }

                    Image img = Image.FromFile(path);

                    RtfTreeNode imgGroup = new RtfTreeNode(RtfNodeType.Group);
                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "pict", false, 0));

                    string format = "";
                    if (path.ToLower().EndsWith("wmf"))
                        format = "emfblip";
                    else
                        format = "jpegblip";

                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, format, false, 0));
                    
                    
                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "picw", true, img.Width * 20));
                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "pich", true, img.Height * 20));
                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "picwgoal", true, width * 20));
                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "pichgoal", true, height * 20));
                    imgGroup.AppendChild(new RtfTreeNode(RtfNodeType.Text, hexdata.ToString(), false, 0));

                    mainGroup.AppendChild(imgGroup);
                }
                finally
                {
                    br.Close();
                    fStream.Close();
                }
            }

            /// <summary>
            /// Establece el formato de caracter por defecto.
            /// </summary>
            public void ResetCharFormat()
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "plain", false, 0));
            }

            /// <summary>
            /// Establece el formato de prrafo por defecto.
            /// </summary>
            public void ResetParFormat()
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "pard", false, 0));
            }

            /// <summary>
            /// Establece el formato de caracter y prrafo por defecto.
            /// </summary>
            public void ResetFormat()
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "pard", false, 0));
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "plain", false, 0));
            }

            /// <summary>
            /// Actualiza los valores de las propiedades de formato de documento.
            /// </summary>
            /// <param name="format">Formato de documento.</param>
            public void UpdateDocFormat(RtfDocumentFormat format)
            {
                docFormat.MarginL = format.MarginL;
                docFormat.MarginR = format.MarginR;
                docFormat.MarginT = format.MarginT;
                docFormat.MarginB = format.MarginB;
            }

            /// <summary>
            /// Actualiza los valores de las propiedades de formato de texto y prrafo.
            /// </summary>
            /// <param name="format">Formato de texto a insertar.</param>
            public void UpdateCharFormat(RtfCharFormat format)
            {
                if (currentFormat != null)
                {
                    SetFormatColor(format.Color);
                    SetFormatSize(format.Size);
                    SetFormatFont(format.Font);

                    SetFormatBold(format.Bold);
                    SetFormatItalic(format.Italic);
                    SetFormatUnderline(format.Underline);
                }
                else //currentFormat == null
                {
                    int indColor = colorTable.IndexOf(format.Color);

                    if (indColor == -1)
                    {
                        colorTable.AddColor(format.Color);
                        indColor = colorTable.IndexOf(format.Color);
                    }

                    int indFont = fontTable.IndexOf(format.Font);

                    if (indFont == -1)
                    {
                        fontTable.AddFont(format.Font);
                        indFont = fontTable.IndexOf(format.Font);
                    }

                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "cf", true, indColor));
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "fs", true, format.Size * 2));
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "f", true, indFont));

                    if (format.Bold)
                        mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "b", false, 0));

                    if (format.Italic)
                        mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "i", false, 0));

                    if (format.Underline)
                        mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "ul", false, 0));

                    currentFormat = new RtfCharFormat();
                    currentFormat.Color = format.Color;
                    currentFormat.Size = format.Size;
                    currentFormat.Font = format.Font;
                    currentFormat.Bold = format.Bold;
                    currentFormat.Italic = format.Italic;
                    currentFormat.Underline = format.Underline;
                }
            }

            /// <summary>
            /// Establece el formato de prrafo pasado como parmetro.
            /// </summary>
            /// <param name="format">Formato de prrafo a utilizar.</param>
            public void UpdateParFormat(RtfParFormat format)
            {
                SetAlignment(format.Alignment);
                SetLeftIndentation(format.LeftIndentation);
                SetRightIndentation(format.RightIndentation);
            }

            /// <summary>
            /// Estable la alineacin del texto dentro del prrafo.
            /// </summary>
            /// <param name="align">Tipo de alineacin.</param>
            public void SetAlignment(TextAlignment align)
            {
                if (currentParFormat.Alignment != align)
                {
                    string keyword = "";

                    switch (align)
                    { 
                        case TextAlignment.Left:
                            keyword = "ql";
                            break;
                        case TextAlignment.Right:
                            keyword = "qr";
                            break;
                        case TextAlignment.Centered:
                            keyword = "qc";
                            break;
                        case TextAlignment.Justified:
                            keyword = "qj";
                            break;
                    }

                    currentParFormat.Alignment = align;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, keyword, false, 0));
                }
            }

            /// <summary>
            /// Establece la sangra izquierda del prrafo.
            /// </summary>
            /// <param name="val">Sangra izquierda en centmetros.</param>
            public void SetLeftIndentation(float val)
            {
                if (currentParFormat.LeftIndentation != val)
                {
                    currentParFormat.LeftIndentation = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "li", true, calcTwips(val)));
                }
            }

            /// <summary>
            /// Establece la sangra derecha del prrafo.
            /// </summary>
            /// <param name="val">Sangra derecha en centmetros.</param>
            public void SetRightIndentation(float val)
            {
                if (currentParFormat.RightIndentation != val)
                {
                    currentParFormat.RightIndentation = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "ri", true, calcTwips(val)));
                }
            }

            /// <summary>
            /// Establece el indicativo de fuente negrita.
            /// </summary>
            /// <param name="val">Indicativo de fuente negrita.</param>
            public void SetFormatBold(bool val)
            {
                if (currentFormat.Bold != val)
                {
                    currentFormat.Bold = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "b", val ? false : true, 0));
                }
            }

            /// <summary>
            /// Establece el indicativo de fuente cursiva.
            /// </summary>
            /// <param name="val">Indicativo de fuente cursiva.</param>
            public void SetFormatItalic(bool val)
            {
                if (currentFormat.Italic != val)
                {
                    currentFormat.Italic = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "i", val ? false : true, 0));
                }
            }

            /// <summary>
            /// Establece el indicativo de fuente subrayada.
            /// </summary>
            /// <param name="val">Indicativo de fuente subrayada.</param>
            public void SetFormatUnderline(bool val)
            {
                if (currentFormat.Underline != val)
                {
                    currentFormat.Underline = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "ul", val ? false : true, 0));
                }
            }

            /// <summary>
            /// Establece el color de fuente actual.
            /// </summary>
            /// <param name="val">Color de la fuente.</param>
            public void SetFormatColor(Color val)
            {
                if (currentFormat.Color.ToArgb() != val.ToArgb())
                {
                    int indColor = colorTable.IndexOf(val);

                    if (indColor == -1)
                    {
                        colorTable.AddColor(val);
                        indColor = colorTable.IndexOf(val);
                    }

                    currentFormat.Color = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "cf", true, indColor));
                }
            }

            /// <summary>
            /// Establece el tamao de fuente actual.
            /// </summary>
            /// <param name="val">Tamao de la fuente.</param>
            public void SetFormatSize(int val)
            {
                if (currentFormat.Size != val)
                {
                    currentFormat.Size = val;

                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "fs", true, val * 2));
                }
            }

            /// <summary>
            /// Establece el tipo de letra actual.
            /// </summary>
            /// <param name="val">Tipo de letra.</param>
            public void SetFormatFont(string val)
            {
                if (currentFormat.Font != val)
                {
                    int indFont = fontTable.IndexOf(val);

                    if (indFont == -1)
                    {
                        fontTable.AddFont(val);
                        indFont = fontTable.IndexOf(val);
                    }

                    currentFormat.Font = val;
                    mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "f", true, indFont));
                }
            }

            #endregion

            #region Metodos Privados

            /// <summary>
            /// Obtiene el cdigo hexadecimal de un entero.
            /// </summary>
            /// <param name="code">Nmero entero.</param>
            /// <returns>Cdigo hexadecimal del entero pasado como parmetro.</returns>
            private string GetHexa(byte code)
            {
                string hexa = Convert.ToString(code, 16);

                if (hexa.Length == 1)
                {
                    hexa = "0" + hexa;
                }

                return hexa;
            }

            /// <summary>
            /// Inserta el cdigo RTF de la tabla de fuentes en el documento.
            /// </summary>
            private void InsertFontTable()
            {
                RtfTreeNode ftGroup = new RtfTreeNode(RtfNodeType.Group);
                
                ftGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "fonttbl", false, 0));

                for(int i=0; i<fontTable.Count; i++)
                {
                    RtfTreeNode ftFont = new RtfTreeNode(RtfNodeType.Group);
                    ftFont.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "f", true, i));
                    ftFont.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "fnil", false, 0));
                    ftFont.AppendChild(new RtfTreeNode(RtfNodeType.Text, fontTable[i] + ";", false, 0));

                    ftGroup.AppendChild(ftFont);
                }

                mainGroup.InsertChild(5, ftGroup);
            }

            /// <summary>
            /// Inserta el cdigo RTF de la tabla de colores en el documento.
            /// </summary>
            private void InsertColorTable()
            {
                RtfTreeNode ctGroup = new RtfTreeNode(RtfNodeType.Group);

                ctGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "colortbl", false, 0));

                for (int i = 0; i < colorTable.Count; i++)
                {
                    ctGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "red", true, colorTable[i].R));
                    ctGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "green", true, colorTable[i].G));
                    ctGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "blue", true, colorTable[i].B));
                    ctGroup.AppendChild(new RtfTreeNode(RtfNodeType.Text, ";", false, 0));
                }

                mainGroup.InsertChild(6, ctGroup);
            }

            /// <summary>
            /// Inserta el cdigo RTF de la aplicacin generadora del documento.
            /// </summary>
            private void InsertGenerator()
            {
                RtfTreeNode genGroup = new RtfTreeNode(RtfNodeType.Group);

                genGroup.AppendChild(new RtfTreeNode(RtfNodeType.Control, "*", false, 0));
                genGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "generator", false, 0));
                genGroup.AppendChild(new RtfTreeNode(RtfNodeType.Text, "NRtfTree Library 1.3.0;", false, 0));

                mainGroup.InsertChild(7, genGroup);
            }

            /// <summary>
            /// Inserta todos los nodos de texto y control necesarios para representar un texto determinado.
            /// </summary>
            /// <param name="text">Texto a insertar.</param>
            private void InsertText(string text)
            {
                int i = 0;
                int code = 0;

                while(i < text.Length)
                {
                    code = Char.ConvertToUtf32(text, i);

                    if(code >= 32 && code < 128)
                    {
                        StringBuilder s = new StringBuilder("");

                        while (i < text.Length && code >= 32 && code < 128)
                        {
                            s.Append(text[i]);

                            i++;

                            if(i < text.Length)
                                code = Char.ConvertToUtf32(text, i);
                        }

                        mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Text, s.ToString(), false, 0));
                    }
                    else
                    {
                        byte[] bytes = encoding.GetBytes(new char[] { text[i] });

                        mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Control, "'", true, bytes[0]));

                        i++;
                    }
                }
            }

            /// <summary>
            /// Actualiza la tabla de fuentes con una nueva fuente si es necesario.
            /// </summary>
            /// <param name="format"></param>
            private void UpdateFontTable(RtfCharFormat format)
            {
                if (fontTable.IndexOf(format.Font) == -1)
                {
                    fontTable.AddFont(format.Font);
                }
            }

            /// <summary>
            /// Actualiza la tabla de colores con un nuevo color si es necesario.
            /// </summary>
            /// <param name="format"></param>
            private void UpdateColorTable(RtfCharFormat format)
            {
                if (colorTable.IndexOf(format.Color) == -1)
                {
                    colorTable.AddColor(format.Color);
                }
            }

            /// <summary>
            /// Inicializa el arbol RTF con todas las claves de la cabecera del documento.
            /// </summary>
            private void InitializeTree()
            {
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "rtf", true, 1));
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "ansi", false, 0));
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "ansicpg", true, encoding.CodePage));
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "deff", true, 0));
                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "deflang", true, CultureInfo.CurrentCulture.LCID));

                mainGroup.AppendChild(new RtfTreeNode(RtfNodeType.Keyword, "pard", false, 0));
            }

            /// <summary>
            /// Inserta las propiedades de formato del documento
            /// </summary>
            private void InsertDocSettings()
            {
                int indInicioTexto = mainGroup.ChildNodes.IndexOf("pard");

                //Generic Properties
                
                mainGroup.InsertChild(indInicioTexto, new RtfTreeNode(RtfNodeType.Keyword, "viewkind", true, 4));
                mainGroup.InsertChild(indInicioTexto++, new RtfTreeNode(RtfNodeType.Keyword, "uc", true, 1));

                //RtfDocumentFormat Properties

                mainGroup.InsertChild(indInicioTexto++, new RtfTreeNode(RtfNodeType.Keyword, "margl", true, calcTwips(docFormat.MarginL)));
                mainGroup.InsertChild(indInicioTexto++, new RtfTreeNode(RtfNodeType.Keyword, "margr", true, calcTwips(docFormat.MarginR)));
                mainGroup.InsertChild(indInicioTexto++, new RtfTreeNode(RtfNodeType.Keyword, "margt", true, calcTwips(docFormat.MarginT)));
                mainGroup.InsertChild(indInicioTexto++, new RtfTreeNode(RtfNodeType.Keyword, "margb", true, calcTwips(docFormat.MarginB)));
            }

            /// <summary>
            /// Convierte entre centmetros y twips.
            /// </summary>
            /// <param name="centimeters">Valor en centmetros.</param>
            /// <returns>Valor en twips.</returns>
            private int calcTwips(float centimeters)
            {
                //1 inches = 2.54 centimeters
                //1 inches = 1440 twips
                //20 twip = 1 pixel

                //X centimetros --> (X*1440)/2.54 twips

                return (int)((centimeters * 1440F) / 2.54F);
            }

            #endregion
        }
    }
}
